extern "C" {
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <tcl.h>
}


#include <QThread>
#include <QObjectList>
#include <QMetaObject>
#include <QPointer>
#include <QHash>

#include "QTclUtil.h"
#include "QTclObject.h"
#include "QTclUI.h"
#include "QTclTimer.h"
#include "QTclFileevent.h"
#include "QTclUtil.h"
#include "QTclQt.h"

static const char *interactive0=
  "set tcl_interactive 0\n"
;

static const char *interactive1=
  "set tcl_interactive 1\n"
  "fileevent stdin readable {\n"
  "if { [ gets stdin line ] < 0 } exit\n"
  "catch { uplevel #0 $line }\n"
  "}\n"
;

static const char *interactive2=
  "set tcl_interactive 1\n"
  "fileevent stdin readable {\n"
  "if { [ gets stdin line ] < 0 } exit\n"
  "catch { uplevel #0 $line } erg\n"
  "if { \"$erg\" != \"\" }  {\n"
  "  puts stdout $erg\n"
  "}\n"
  "puts -nonewline \"%1 % \"; flush stdout\n"
  "}\n"
  "puts -nonewline \"%1 % \"; flush stdout\n"
;

#ifdef TESTS
static const char *interactive1xx=
  "set tcl_interactive 1\n"
  "set halfLine {}\n"
  //"fconfigure stdin -buffering line -blocking 0\n"
  "fileevent stdin readable {\n"
  "  if [ eof stdin ] exit\n"
  "  while 1 {\n"
  "    if { [ gets stdin line ] < 0 } {\n"
  "      set halfLine $line\n"
  "      break\n"
  "    }\n"
  "    append halfLine $line\n"
  "    if [ catch { uplevel #0 $halfLine } erg ] {\n"
  "      puts stderr $erg\n"
  "    }\n"
  "    set halfLine {}\n"
  "  }\n"
  "  flush stdout\n"
  "}\n"
;

static const char *interactive1xy=
  "set tcl_interactive 1\n"
  "fileevent stdin readable {\n"
  "if { [ gets stdin line ] < 0 } exit\n"
  "if [ catch { uplevel #0 $line } erg ] {\n"
  "  puts stdout $erg\n"
  "} else {\n"
  "  puts stdout $erg\n"
  "}\n"
  "flush stdout\n"
  "}\n"
;

#endif

static const char *createConsole=
  "proc console_status { what msg } {\n"
  "  qt {.console.automatic status bar} message $what:$msg\n"
  "}\n"
  "proc bgerror { msg } { \n"
  "  console_status BGERROR $msg\n"
  "}\n"
  "qt create QMainWindow .console\n"
  "qt create QWidget .console.f\n"
  "qt create QComboBox .console.f.e editable true duplicatesEnabled false\n"
  "qt create QTextEdit .console.f.t focusPolicy 0 readOnly true\n"
  "qt create QGridLayout .console.f.g\n"
  "qt .console.f.g add .console.f.t row 0 column 0\n"
  "qt .console.f.g add .console.f.e row 1 column 0\n"
  "qt {.console.f.e.combo edit} bind returnPressed() {\n"
  "  set _text [qt .console.f.e cget currentText ] \n"
  "  qt .console.f.t append \"%% $_text\"\n"
  "  if {[ catch { uplevel #0 $_text } _result ] } {\n"
  "    qt .console.f.t append $_result\n"
  "    console_status ERROR \"\"\n"
  "  } else {\n"
  "    if { [ string length $_result ] } {\n"
  "      qt .console.f.t append $_result\n"
  "    }\n"
  "    console_status OK \"\"\n"
  "  }\n"
  "  qt .console.f.t scrollToBottom\n"
  "  qt {.console.f.e.combo edit} selectAll\n"
  "  unset _text _result\n"
  "}\n"
  "after -200 catch { update idletasks }\n"
  "qt .console resize 600 400\n"
  "qt .console centralWidget .console.f\n"
  "qt .console show\n"
;

static const char *removeConsole=
  "rename console_status \"\"\n"
  "rename bgerror \"\"\n"
  "qt .console destroy\n"
;

QTclInterp::QTclInterp(QObject *parent):QObject(parent),_debug(0),_interactive(0),_interp(Tcl_CreateInterp()) {
  QTclInitQt(this);
}

void QTclInterp::setInteractive(int interactive) {
  if(interactive==_interactive) return;
  switch(interactive) {
    default:
      return;
    case 0: Tcl_GlobalEval(_interp,(char*)interactive0);
      break;
    case 1:
      if(interactive) Tcl_GlobalEval(_interp,(char*)interactive0);
      Tcl_GlobalEval(_interp,(char*)interactive1);
      break;
    case 2:
      if(interactive) Tcl_GlobalEval(_interp,(char*)interactive0);
      Tcl_GlobalEval(_interp,(char*)interactive2);
      break;
  }
  _interactive=interactive;
}

void QTclInterp::unsetInteractive() {
  setInteractive(0);
}

void QTclInterp::setDebug(bool debug) {
  if(debug==_debug) return;
  switch(debug) {
    default: // never happens
      return;
    case 0:
      Tcl_GlobalEval(_interp,(char*)removeConsole);
      break;
    case 1:
      Tcl_GlobalEval(_interp,(char*)createConsole);
      break;
  }
  _debug=debug;
}

void QTclInterp::unsetDebug() {
  setDebug(FALSE);
}

void QTclInterp::setDebug() {
  setDebug(TRUE);
}

int QTclInterp::eval(const QString&cmd) {
  return Tcl_GlobalEval(_interp,QTclS2C(cmd));
}

int QTclInterp::evalFile(const QString& fileName) {
  return Tcl_EvalFile(_interp,QTclS2C(fileName));
}

QString QTclInterp::result(void) const {
  return Tcl_GetStringResult(_interp);
}

QTclInterp::~QTclInterp() {
  Tcl_DeleteInterp(_interp);
}

static QList<QObject *>rootObjects;

void QTclInterp::appendRootObjects(Tcl_Interp *interp) {
  for(QObjectList::const_iterator i=rootObjects.begin();i!=rootObjects.end();++i) {
    Tcl_AppendElement(interp,QTclS2C(name(*i)));
  }
}

class QTclDeleter : public QTclCommand {
public:
  QTclDeleter(QObject *parent,const QString &name,Tcl_Interp *interp,const QString &command) : QTclCommand(parent,name,interp,command) {
    if(!parent->parent()) {
      rootObjects.append(parent);
    }
  }
  ~QTclDeleter() {
     Tcl_DeleteCommand(interp(),QTclS2C(command()));
     rootObjects.removeAll(parent());
  }
};

void QTclInterp::registerObject(Tcl_Interp *interp,QObject *w,int deep) {
QString name=QTclInterp::name(w);
  Tcl_CmdProc *cmdProc=QTclInterp::findMethods(w);
  Tcl_CreateCommand(interp,QTclS2C(name),cmdProc,(ClientData)w,0);
  new QTclDeleter(w,"deleter",interp,name);
  if(!deep) return;
  const QObjectList &cl=w->children();
  for(QObjectList::const_iterator i=cl.begin();i!=cl.end();++i) {
    registerObject(interp,*i,deep);
  }
}

static QString pointer2string(const QObject*o) {
qulonglong v=(qulonglong)o;
  return QString("O0x%1").arg(v,0,16);
}

static QObject* string2pointer(const QString &s) {
bool ok;
  if(s.at(0)!='O') return 0;
  qulonglong v=s.mid(1).toULongLong(&ok,16);
  if(!ok) return 0;
  return (QObject*)v;
}

static QObject*findObjectByName(Tcl_Interp *interp,const QString &path) {
QObject *o=0;
Tcl_CmdInfo info;
  QStringList p=path.split('.');
  if((p.size()<2) || !(p[0].isEmpty())) return 0;
  if((!Tcl_GetCommandInfo(interp,QTclS2C(QString(".")+p[1]),&info)) || (!(o=(QObject*)(info.clientData)))) {
    o=string2pointer(p[1]);
    if(!o) {
      return 0;
    }
  }
  for(int i=2;i<p.size();i++) {
    QString n=p[i];
    QObject *child=o->findChild<QObject*>(p[i]);
    if(!child) {
      if(!(child=string2pointer(n))) {
        return 0;
      }
      if(!o->children().contains(child)) {
        return 0;
      }
    }
    o=child;
  }
  return o;
}


QString QTclInterp::name(const QObject *o) {
QString result;
  for(;o;o=o->parent()) {
    QString n=o->objectName();
    if(n.isEmpty()) {
      n=pointer2string(o);
    }
    result=result.prepend(n);
    result=result.prepend('.');
  }
  return result;
}

int QTclInterp::getObjectByName(Tcl_Interp *interp,const QString& name,QObject **result,const char *className) {
  *result=0;
  QObject *o=findObjectByName(interp,name);
  if(!o) {
    Tcl_AppendResult(interp,"object '",QTclS2C(name),"' not found",0);
    return TCL_ERROR;
  }
  if(className && !(o->inherits(className))) {
    Tcl_AppendResult(interp,"object '",QTclS2C(name),"' is not a '",className,"'",0);
    return TCL_ERROR;
  }
  *result=o;
  return TCL_OK;
}

int QTclInterp::getWidgetByName(Tcl_Interp *interp,const QString &name,QWidget **result) {
QObject *o;
  if(getObjectByName(interp,name,&o,"QWidget")!=TCL_OK) return TCL_ERROR;
  *result=(QWidget*)o;
  return TCL_OK;
}

void QTclInterp::appendName(Tcl_Interp *interp,const QObject *o) {
  if(o) {
    Tcl_AppendElement(interp,QTclS2C(name(o)));
  } else {
    Tcl_AppendElement(interp,"");
  }
}

void QTclInterp::appendChildren(Tcl_Interp *interp,const QObject *w,int deepMode,const char *className) {
const QObjectList &cl=w->children();
  for(QObjectList::const_iterator i=cl.begin();i!=cl.end();++i) {
    QObject *o=*i;
    if(className && !o->inherits(className)) {
      continue; 
    }
    Tcl_AppendElement(interp,QTclS2C(name(o)));
    if(deepMode) {
      appendChildren(interp,o,deepMode,className);
    }
  }
}

static QHash<QString,Tcl_CmdProc*> QTclMethods;

int QTclInterp::registerMethods(const char *className,Tcl_CmdProc *cmdProc) {
  QTclMethods.insert(className,cmdProc);
  return TCL_OK;
}

Tcl_CmdProc *QTclInterp::findMethods(const QObject *o) {
  for(const QMetaObject *m=o->metaObject();m;m=m->superClass()) {
    const char *className=m->className();
    Tcl_CmdProc *p=QTclMethods.value(className);
    if(p) return p;
  }
  return QTclObjectMethods; // can't happen
}

static QHash<QString,Tcl_CmdProc*> QTclClassMethods;

int QTclInterp::registerClassMethods(const char *className,Tcl_CmdProc *cmdProc) {
  QTclClassMethods.insert(className,cmdProc);
  return TCL_OK;
}

Tcl_CmdProc *QTclInterp::findClassMethods(const char *className) {
  return QTclClassMethods.value(className);
}

Tcl_CmdProc *QTclInterp::findClassMethods(const QObject *o) {
  for(const QMetaObject *m=o->metaObject();m;m=m->superClass()) {
    const char *className=m->className();
    Tcl_CmdProc *p=QTclMethods.value(className);
    if(p) return p;
  }
  return QTclObjectClassMethods;
}

void QTclInterp::appendClassMethodNames(Tcl_Interp *interp) {
  for(QHash<QString,Tcl_CmdProc*>::const_iterator i=QTclClassMethods.begin();i!=QTclClassMethods.end();++i) {
    Tcl_AppendElement(interp,QTclS2C(i.key()));
  }
}

/* ----------------------------- globalMethods ------------------------- */

static QHash<QString,Tcl_CmdProc*> QTclGlobalMethods;

int QTclInterp::registerGlobalMethod(const char *name,Tcl_CmdProc *cmdProc) {
  QTclGlobalMethods.insert(name,cmdProc);
  return TCL_OK;
}

Tcl_CmdProc *QTclInterp::findGlobalMethod(const char *name) {
  return QTclGlobalMethods.value(name);
}

void QTclInterp::appendGlobalMethodNames(Tcl_Interp *interp) {
  for(QHash<QString,Tcl_CmdProc*>::const_iterator i=QTclGlobalMethods.begin();i!=QTclGlobalMethods.end();++i) {
    Tcl_AppendElement(interp,QTclS2C(i.key()));
  }
}

/* ----------------------------- GlobalFactory ------------------------- */

class QTclGlobalFactory {
public:
  static void addFactory(QTclFactory *factory) {
    factoryList.append(factory);
  }
  static QObject*create(QString className,QObject *parent,const QString &name) {
    for(int i=0;i<(int)factoryList.count();i++) {
      QTclFactory *f=factoryList.at(i);
      QObject *result=f->create(className,parent);
      if(result) {
        result->setObjectName(name);
        return result;
      }
    }
    return 0;
  }
  static QStringList classNames() {
    QStringList result;
    for(int i=0;i<(int)factoryList.count();i++) {
      QTclFactory *f=factoryList.at(i);
      result+=f->classNames();
    }
    return result;
  }
private:
  static QList<QTclFactory*> factoryList;
};

QList<QTclFactory*> QTclGlobalFactory::factoryList;

int QTclInterp::registerFactory(QTclFactory*factory) {
  QTclGlobalFactory::addFactory(factory);
  return TCL_OK;
}

void QTclInterp::appendClassNames(Tcl_Interp *interp) {
  Tcl_AppendQStringList(interp,QTclGlobalFactory::classNames());
}

QObject *QTclInterp::create(const char *className,QObject *parent,const QString &name) {
  return QTclGlobalFactory::create(className,parent,name); 
}

/* ----------------------------- Factory end ------------------------- */

int QTclInterp::eval(Tcl_Interp *interp,const QString& command) {
int result;
  if((result=Tcl_GlobalEval(interp,QTclS2C(command)))!=TCL_OK) {
    Tcl_BackgroundError(interp);
  }
  return result;
}

int QTclInterp::eval(Tcl_Interp *interp,const QString& command,const QString& arg1) {
QString cmd=command.arg(arg1);
  return eval(interp,cmd);
}

int QTclInterp::eval(Tcl_Interp *interp,const QString& command,const QString& arg1,const QString &arg2) {
QString cmd=command.arg(arg1).arg(arg2);
  return eval(interp,cmd);
}

int QTclInterp::eval(Tcl_Interp *interp,const QString& command,const QString& arg1,const QString &arg2,const QString &arg3) {
QString cmd=command.arg(arg1).arg(arg2).arg(arg3);
  return eval(interp,cmd);
}

int QTclInterp::prepareForCreate(Tcl_Interp *interp,const QString& path,const char *className,QObject **parent,QString &name) {
  if(findObjectByName(interp,path)) {
    Tcl_AppendResult(interp,"object exists '",QTclS2C(path),"'",0);
    return TCL_ERROR;
  }
  QString parentName=path.section('.',0,-2);
  name=path.section('.',-1);
  if(name.isEmpty()) {
    Tcl_AppendResult(interp,"invalid object name:'",QTclS2C(path),"'",0);
    return TCL_ERROR;
  }
  *parent=0;
  if(!parentName.isEmpty()) {
    if(getObjectByName(interp,parentName,parent,className)!=TCL_OK) {
      Tcl_AppendResult(interp,"parent '",QTclS2C(parentName),"' does not exist",0);
      return TCL_ERROR;
    }
  }
  return TCL_OK;
}

#ifdef LATER
static int scanOptions(Tcl_Interp *interp,int argc,char *argv[],int offset,int size,Options *options,...) {
va_list v;
  void *results[10]; va_start(v,options);
  for(int i=0;i<size && i<QTCLARSIZE(results);i++) {
    argv[i]=va_arg(v,void *);
  }
  va_end(v);
  for(i=offset,i<argc;i++) {
  }
}
#endif

#include "moc_QTclInterp.cpp"
