extern "C" {
#include <stdlib.h>
#include <tcl.h>
}

#include <Qt>
#include <QDate>
#include <QTime>
#include <QMetaProperty>
#include <QSizePolicy>
#include <QRegion>
#include <QFont>
#include <QBrush>
#include <QPen>
#include <QValidator>
#include <QKeySequence>

#include "QTclObject.h"
#include "QTclPixmap.h"
#include "QTclUtil.h"
#include "QTclBind.h"
#include "QTclEvent.h"
#include "QTclInterp.h"
#include "QTclSizePolicy.h"

static int scanSelf(Tcl_Interp *interp,int argc,char *argv[],int *mode) {
 *mode=TRUE;
  switch(argc) {
    case 2:
      return TCL_OK;
    case 3:
      if(!strcmp(argv[2],"-self")) {
	*mode=FALSE;
        return TCL_OK;
      }
    default:
      return Tcl_WrongArgs(interp,2,argv,"[-self]");
  }
}

static int QTclSetProperty(Tcl_Interp *interp,QObject *w,const char *propertyName,const QString &value) {
const QMetaObject*m=w->metaObject();
int prop=m->indexOfProperty(propertyName);
  if(!m) {
    Tcl_AppendResult(interp,"property '",propertyName,"' not found",0);
    return TCL_ERROR;
  }
  QMetaProperty p=m->property(prop);
  if(!p.isWritable()) {
    Tcl_AppendResult(interp,"property '",propertyName,"' not writable",0);
    return TCL_ERROR;
  }
  const char *type=p.typeName();
  bool result=false;
  if(!strcmp(type,"QString")) {
    result=w->setProperty(propertyName,QVariant(value));
  } else if(!strcmp(type,"QCString")) {
    result=w->setProperty(propertyName,QVariant(value));
  } else if(p.isEnumType()) {
    result=w->setProperty(propertyName,QVariant(value));
  } else if(p.isFlagType()) {
    int largc;
    char **largv;
    if(Tcl_SplitList(interp,(char*)QTclS2C(value),&largc,&largv)!=TCL_OK) return TCL_ERROR;
    QStringList l;
    for(int i=0;i<largc;i++) {
      l.append(QString(largv[i]));
    }
    Tcl_Free((char*)largv);
    //ckfree((char*)largv);
    result=w->setProperty(propertyName,QVariant(value)); 
  } else if(!strcmp(type,"bool")) {
    int bval;
    if(Tcl_GetBoolean(interp,QTclS2C(value),&bval)!=TCL_OK) return TCL_ERROR;
    result=w->setProperty(propertyName,QVariant((bool)bval));
  } else if(!strcmp(type,"int") || !strcmp(type,"uint")) {
    int ival;
    if(Tcl_GetInt(interp,QTclS2C(value),&ival)!=TCL_OK) return TCL_ERROR;
    result=w->setProperty(propertyName,QVariant(ival));
  } else if(!strcmp(type,"double")) {
    double dval;
    if(Tcl_GetDouble(interp,QTclS2C(value),&dval)!=TCL_OK) return TCL_ERROR;
    result=w->setProperty(propertyName,QVariant(dval));
  } else if(!strcmp(type,"QCursor")) {
    int ival;
    if(Tcl_GetInt(interp,QTclS2C(value),&ival)!=TCL_OK) {
      Tcl_AppendResult(interp,"only cursor integer implemented",0);
      return TCL_ERROR;
    }
    result=w->setProperty(propertyName,QVariant(ival)); 
  } else if(!strcmp(type,"QColor")) {
    result=w->setProperty(propertyName,QVariant(QColor(value)));
  } else if(!strcmp(type,"QFont")) {
    QFont f;
    if(!f.fromString(value)) {
      Tcl_AppendResult(interp,"invalid font value '",QTclS2C(value),"'",0);
      return TCL_ERROR;
    }
    result=w->setProperty(propertyName,QVariant(f));
  } else if(!strcmp(type,"QPixmap")) {
    if(value.isEmpty()) {
      result=w->setProperty(propertyName,QVariant(QPixmap()));
    } else {
      QPixmap pixmap;
      if(QTclGetPixmap(interp,value,&pixmap)!=TCL_OK) return TCL_ERROR;
      result=w->setProperty(propertyName,QVariant(pixmap));
    }
  } else if(!strcmp(type,"QImage")) {
    result=w->setProperty(propertyName,QVariant(QImage(QString(value))));
  } else if(!strcmp(type,"QRect")) {
    int x;
    int y;
    int width;
    int height;
    if(sscanf(QTclS2C(value),"%d %d %d %d",&x,&y,&width,&height)!=4) {
      Tcl_AppendResult(interp,"No valid Rectange '",QTclS2C(value),"'",0); 
      return TCL_ERROR;
    }
    QRect r(x,y,width,height);
    result=w->setProperty(propertyName,QVariant(r));
  } else if(!strcmp(type,"QRegion")) {
    int x;
    int y;
    int width;
    int height;
    if(sscanf(QTclS2C(value),"%d %d %d %d",&x,&y,&width,&height)!=4) {
      Tcl_AppendResult(interp,"No valid Region '",QTclS2C(value),"'",0); 
      return TCL_ERROR;
    }
    result=w->setProperty(propertyName,QVariant(QRegion(x,y,width,height)));
  } else if(!strcmp(type,"QSize")) {
    int width;
    int height;
    if(sscanf(QTclS2C(value),"%d %d",&width,&height)!=2) {
      Tcl_AppendResult(interp,"No valid QSize '",QTclS2C(value),"'",0); 
      return TCL_ERROR;
    }
    result=w->setProperty(propertyName,QVariant(QSize(width,height)));
  } else if(!strcmp(type,"QSizePolicy")) {
    QSizePolicy s;
    if(QTclGetSizePolicy(interp,value,&s)!=TCL_OK) return TCL_ERROR;
    result=w->setProperty(propertyName,QVariant(s));
  } else if(!strcmp(type,"QPoint")) {
    int x;
    int y;
    if(sscanf(QTclS2C(value),"%d %d",&x,&y)!=2) {
      Tcl_AppendResult(interp,"No valid QPoint '",QTclS2C(value),"'",0); 
      return TCL_ERROR;
    }
    result=w->setProperty(propertyName,QVariant(QPoint(x,y)));
  } else if(!strcmp(type,"QDate")) {
    result=w->setProperty(propertyName,QVariant(QDate::fromString(value)));
  } else if(!strcmp(type,"QTime")) {
    result=w->setProperty(propertyName,QVariant(QTime::fromString(value)));
  } else if(!strcmp(type,"QDateTime")) {
    result=w->setProperty(propertyName,QVariant(QDateTime::fromString(value)));
  } else if(!strcmp(type,"QKeySequence")) {
    QVariant v;
    v.setValue(QKeySequence(value));
    result=w->setProperty(propertyName,v);
  } else if(!strcmp(type,"QStringList")) {
    QStringList s;
    int largc;
    char **largv;
    if(Tcl_SplitList(interp,(char*)QTclS2C(value),&largc,&largv)!=TCL_OK) return TCL_ERROR;
    for(int i=0;i<largc;i++) {
      s.append(largv[i]);
    }
    Tcl_Free((char*)largv);
    //ckfree((char*)largv);
    result=w->setProperty(propertyName,QVariant(s));
  } else {
    Tcl_AppendResult(interp,"property set for type '",type,"' not implemented",0);
    return TCL_ERROR;
  }
  if(!result) {
    Tcl_AppendResult(interp,"property set of '",type,"' not succeeded",0);
    return TCL_ERROR;
  }
  return TCL_OK;
}

static int QTclAppendQVariant(Tcl_Interp *interp,const QVariant &v,const QMetaProperty *propPtr,int quoted) {
  switch(v.type()) {
    case QVariant::Invalid:
      Tcl_AppendElement(interp,""); // fix later
      return TCL_OK;
    case QVariant::Map: {
       Tcl_AppendResult(interp," {",0);
        for(QMap<QString,QVariant>::const_iterator i=v.toMap().begin();i!=v.toMap().end();++i) {
          Tcl_AppendResult(interp,"{",0);
          Tcl_AppendElement(interp,QTclS2C(i.key()));
          QTclAppendQVariant(interp,i.value(),0,1);
          Tcl_AppendResult(interp,"}",0);
        }
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::List: {
        Tcl_AppendResult(interp," {",0);
        for(QList<QVariant>::const_iterator i=v.toList().begin();i!=v.toList().end();++i) {
          QTclAppendQVariant(interp,*i,0,1);
        }
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Char:
      Tcl_AppendResult(interp,"'",v.toChar().toLatin1(),"'",0);
      break;
    case QVariant::String:
      if(quoted) {
        Tcl_AppendElement(interp,QTclS2C(v.toString()));
      } else {
        Tcl_AppendResult(interp,QTclS2C(v.toString()),0);
      }
      break;
    case QVariant::StringList: {
      Tcl_AppendResult(interp," {",0);
      const QStringList &l=v.toStringList();
      for(QStringList::const_iterator i=l.begin();i!=l.end();++i) {
        Tcl_AppendElement(interp,QTclS2C(*i));
      }
      Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Font:
      Tcl_AppendElement(interp,QTclS2C(v.value<QFont>().toString()));
      break;
    case QVariant::Pixmap: {
        const QPixmap p=v.value<QPixmap>();
        QTclAppendPixmap(interp,p);
      }
      break;
    case QVariant::Brush:
      {
        QBrush b=v.value<QBrush>();
        Tcl_AppendResult(interp," {",0);
        Tcl_AppendInt(interp,b.style());
        Tcl_AppendElement(interp,QTclS2C(b.color().name()));
        QPixmap p=b.texture();
        QTclAppendPixmap(interp,p);
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Pen:
      {
        QPen p=v.value<QPen>();
        Tcl_AppendResult(interp," {",0);
        Tcl_AppendElement(interp,QTclS2C(p.color().name()));
        Tcl_AppendInt(interp,p.width());
        Tcl_AppendInt(interp,p.style());
        Tcl_AppendInt(interp,p.capStyle());
        Tcl_AppendInt(interp,p.joinStyle());
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Rect: 
      {
        Tcl_AppendResult(interp," {",0);
        QRect r=v.toRect();
        Tcl_AppendInt(interp,r.x());
        Tcl_AppendInt(interp,r.y());
        Tcl_AppendInt(interp,r.width());
        Tcl_AppendInt(interp,r.height());
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Size:
      {
        Tcl_AppendResult(interp," {",0);
        QSize s=v.toSize();
        Tcl_AppendInt(interp,s.width());
        Tcl_AppendInt(interp,s.height());
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Color:
      Tcl_AppendElement(interp,QTclS2C(v.value<QColor>().name()));
      break;
    case QVariant::Palette:
      Tcl_AppendResult(interp," {[Palette]}",0);
      break;
    case QVariant::Point:
      {
        QPoint p=v.toPoint();
        Tcl_AppendResult(interp," {",0);
        Tcl_AppendInt(interp,p.x());
        Tcl_AppendInt(interp,p.y());
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Image: {
        QImage i=v.value<QImage>();
        Tcl_AppendResult(interp," {",0);
        Tcl_AppendResult(interp,"[Image]",0);
        Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::Int:
    case QVariant::UInt:
      if(propPtr && propPtr->isEnumType()) {
        unsigned int u=v.toUInt();
        QMetaEnum e=propPtr->enumerator();
        if(propPtr->isFlagType()) {
          QByteArray l=e.valueToKeys(u);
          Tcl_AppendResult(interp,"{",0);
          Tcl_AppendResult(interp,l.data(),0);
          Tcl_AppendResult(interp,"}",0);
        } else {
          Tcl_AppendElement(interp,e.valueToKey(u));
        }
      } else {
        if(v.type()==QVariant::Int) {
          Tcl_AppendInt(interp,v.toInt());
        } else {
          Tcl_AppendIntAsHex(interp,v.toUInt());
        }
      }
      break;
    case QVariant::Bool:
      Tcl_AppendBoolString(interp,v.toBool());
      break;
    case QVariant::Double:
      Tcl_AppendDouble(interp,v.toDouble());
      break;
    case QVariant::SizePolicy:
      Tcl_AppendResult(interp," {",0);
      QTclAppendSizePolicy(interp,v.value<QSizePolicy>());
      Tcl_AppendResult(interp,"}",0);
      break;
    case QVariant::Date: {
        Tcl_AppendElement(interp,QTclS2C(v.toDate().toString()));
      }
      break;
    case QVariant::Time: {
        Tcl_AppendElement(interp,QTclS2C(v.toTime().toString()));
      }
      break;
    case QVariant::DateTime: {
        Tcl_AppendElement(interp,QTclS2C(v.toDateTime().toString()));
      }
      break;
    case QVariant::ByteArray: {
         Tcl_AppendResult(interp," {",0);
         for(QByteArray::ConstIterator x=v.toByteArray().begin();x;x++) {
           Tcl_AppendHexByte(interp,*x);
         }
         Tcl_AppendResult(interp,"}",0);
      }
      break;
    case QVariant::KeySequence:
      Tcl_AppendResult(interp," {",0);
      Tcl_AppendResult(interp,QTclS2C((QString)(v.value<QKeySequence>())),0);
      Tcl_AppendResult(interp,"}",0);
      break;
    case QVariant::LongLong:
      {
        char buffer[100];
        sprintf(buffer,"%Ld",v.toLongLong());
        Tcl_AppendElement(interp,buffer);
      }
      break;
    case QVariant::ULongLong:
      {
        char buffer[100];
        sprintf(buffer,"0x%Lx",v.toULongLong());
        Tcl_AppendElement(interp,buffer);
      }
      break;
    case QVariant::BitArray:
    case QVariant::Url:
    case QVariant::Locale:
    case QVariant::RectF:
    case QVariant::SizeF:
    case QVariant::Line:
    case QVariant::LineF:
    case QVariant::PointF:
    case QVariant::Icon:
    case QVariant::Polygon:
    case QVariant::Region:
    case QVariant::Bitmap:
    case QVariant::Cursor:
    case QVariant::TextLength:
    case QVariant::TextFormat:
    case QVariant::RegExp:
    case QVariant::UserType:
    case QVariant::LastType:
      Tcl_AppendElement(interp,QTclS2C(v.toString()));
      break;
  }
  return TCL_OK;
}

static int QTclAppendPropertyValue(Tcl_Interp *interp,QObject *w,const char *propertyName,int quoted=0) {
  int propId=w->metaObject()->indexOfProperty(propertyName);
  if(propId<0) {
    Tcl_AppendResult(interp,"property not found '",propertyName,"'",0);
    return TCL_ERROR;
  }
  QMetaProperty p=w->metaObject()->property(propId);
  QVariant v=w->property(propertyName);
  return QTclAppendQVariant(interp,v,&p,quoted);
}

static int QTclAppendMetaPropery(Tcl_Interp *interp,QObject *w,const QMetaProperty *propPtr) {
Tcl_DString d;
  Tcl_DStringInit(&d);
  if(propPtr->isWritable()) Tcl_DStringAppendElement(&d,"isWritable");
  if(propPtr->isDesignable(w)) Tcl_DStringAppendElement(&d,"isDesignable");
  if(propPtr->isScriptable(w)) Tcl_DStringAppendElement(&d,"isScriptable");
  if(propPtr->isStored(w))  Tcl_DStringAppendElement(&d,"isStored");
  if(propPtr->isEnumType()) Tcl_DStringAppendElement(&d,"isEnumType");
  if(propPtr->isFlagType()) Tcl_DStringAppendElement(&d,"isFlagType");
  Tcl_AppendElement(interp,Tcl_DStringValue(&d));
  Tcl_DStringFree(&d);
  return TCL_OK;
}

static int QTclAppendTypeInfo(Tcl_Interp *interp,const QMetaProperty *propPtr) {
  if(propPtr->isEnumType()) {
    Tcl_DString d;
    Tcl_DStringInit(&d);
    Tcl_DStringAppend(&d,propPtr->typeName(),-1);
    Tcl_DStringAppend(&d,":",-1);
    QMetaEnum m=propPtr->enumerator();
    for(int i=0;i<m.keyCount();i++) {
      Tcl_DStringAppendElement(&d,m.key(i));
    }
    Tcl_AppendElement(interp,Tcl_DStringValue(&d));
    Tcl_DStringFree(&d);
  } else {
    Tcl_AppendElement(interp,propPtr->typeName());
  }
  return TCL_OK;
}

static int QTclAppendPropery(Tcl_Interp *interp,QObject *w,const char *propertyName) {
QVariant v=w->property(propertyName);
int propId=w->metaObject()->indexOfProperty(propertyName);
  if(propId<0) {
    Tcl_AppendResult(interp,"property not found '",propertyName,"'",0);
    return TCL_ERROR;
  }
  const QMetaProperty p=w->metaObject()->property(propId);
  Tcl_AppendResult(interp,"{",0);
  Tcl_AppendElement(interp,propertyName);
  QTclAppendTypeInfo(interp,&p);
  QTclAppendMetaPropery(interp,w,&p);
  QTclAppendQVariant(interp,v,&p,1);
  Tcl_AppendResult(interp,"}\n",0);
  return TCL_OK;
}

static int QTclAppendProperties(Tcl_Interp *interp,QObject *w) {
const QMetaObject* m=w->metaObject();
  Tcl_AppendResult(interp,"{\n",0);
  for(int i=0;i<m->propertyCount();i++) {
    QMetaProperty p=m->property(i);
    QTclAppendPropery(interp,w,p.name());
  }
  Tcl_AppendResult(interp,"}",0);
  return TCL_OK;
}

int QTclSetProperties(Tcl_Interp *interp,QObject *w,int argc,char *argv[]) {
  if(argc&1) {
    return Tcl_WrongArgs(interp,0,0,"[<property> <value>]..");
  }
  for(;argc;argc-=2,argv+=2) {
    if(QTclSetProperty(interp,w,argv[0],QTclC2S(argv[1]))!=TCL_OK) return TCL_ERROR;
  }
  return TCL_OK;
}

static int QTclInvokeSlot(Tcl_Interp *interp,QObject *w,const char *slotName,int argc,char *argv[]) {
bool slotMatched=false;
const QMetaObject *m=w->metaObject();
  for(int i=0;i<m->methodCount();i++) {
    const QMetaMethod p=m->method(i);
    if(p.methodType()!=QMetaMethod::Slot) continue;
    const char *signature=p.signature();
    const char *kpos=strchr(signature,'(');
    if(!strncmp(slotName,signature,kpos-signature)) {
      slotMatched=true;
      const char *argumentSignature=kpos;
      if(argc==0 && !strcmp(argumentSignature,"()")) {
        QMetaObject::invokeMethod(w,slotName);
        return TCL_OK;
      } else if(!strcmp(argumentSignature,"(int)")) {
        int arg0;
        if(argc!=1) return Tcl_WrongArgs(interp,1,argv,"<int>");
        if(Tcl_GetInt(interp,argv[0],&arg0)!=TCL_OK) return TCL_ERROR;
        QMetaObject::invokeMethod(w,slotName,Q_ARG(int,arg0));
        return TCL_OK;
      } else if(!strcmp(argumentSignature,"(bool)")) {
        int arg0;
        if(argc!=1) return Tcl_WrongArgs(interp,1,argv,"<bool>");
        if(Tcl_GetBoolean(interp,argv[0],&arg0)!=TCL_OK) return TCL_ERROR;
        QMetaObject::invokeMethod(w,slotName,Q_ARG(bool,arg0));
        return TCL_OK;
      } else if(!strcmp(argumentSignature,"(double)")) {
        double arg0;
        if(argc!=1) return Tcl_WrongArgs(interp,1,argv,"<double>");
        if(Tcl_GetDouble(interp,argv[0],&arg0)!=TCL_OK) return TCL_ERROR;
        QMetaObject::invokeMethod(w,slotName,Q_ARG(double,arg0));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(QString)")) {
        if(argc!=1) return Tcl_WrongArgs(interp,1,argv,"<string>");
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QString,QString(argv[0])));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(const QString&)")) {
        if(argc!=1) return Tcl_WrongArgs(interp,1,argv,"<string>");
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QString,QString(argv[0])));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(QString)")) {
        if(argc!=1) return Tcl_WrongArgs(interp,1,argv,"<string>");
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QString,QString(argv[0])));
        return TCL_OK;
      } else if(argc==2 && !strcmp(argumentSignature,"(const QString&,int)")) {
        int arg1;
        if(Tcl_GetInt(interp,argv[1],&arg1)!=TCL_OK) {
          return Tcl_WrongArgs(interp,1,argv,"Error invoking slot(const QString&,int)");
        }
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QString,QString(argv[0])),Q_ARG(int,arg1));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(const QValidator*)")) {
        QObject *object;
        if(QTclInterp::getObjectByName(interp,argv[0],&object,"QValidator")!=TCL_OK) return TCL_ERROR;
        QMetaObject::invokeMethod(w,slotName,Q_ARG(const QValidator*,(const QValidator*)object));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(const QPixmap&)")) {
        if(argc!=1) {
          return Tcl_WrongArgs(interp,1,argv,"Error invoking slot(const QPixmap&)");
        }
        QPixmap pixmap;
        if(QTclGetPixmap(interp,argv[0],&pixmap)!=TCL_OK) return TCL_ERROR;
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QPixmap,pixmap));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(QWidget*)")) {
        QWidget *widget;
        if(QTclInterp::getWidgetByName(interp,argv[0],&widget)!=TCL_OK) return TCL_ERROR;
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QWidget*,widget));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(QObject*)")) {
        if(argc!=1) {
          return Tcl_WrongArgs(interp,1,argv,"Error invoking slot(QObject*)");
        }
        QObject *object;
        if(QTclInterp::getObjectByName(interp,argv[0],&object)!=TCL_OK) return TCL_ERROR;
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QObject*,object));
        return TCL_OK;
      } else if(argc==1 && !strcmp(argumentSignature,"(const QColor&)")) {
        if(argc!=1) {
          return Tcl_WrongArgs(interp,1,argv,"Error invoking slot(const QColor&)");
        }
        QColor c(argv[0]);
        QMetaObject::invokeMethod(w,slotName,Q_ARG(QColor,c));
        return TCL_OK;
      } else if(argc==2 && !strcmp(argumentSignature,"(int,int)")) {
        int arg0,arg1;
        if((argc!=2)
           || (Tcl_GetInt(interp,argv[0],&arg0)!=TCL_OK)
           || (Tcl_GetInt(interp,argv[1],&arg1)!=TCL_OK)
          ) {
          Tcl_AppendResult(interp,"Error invoking slot(int,int)",0);
          return TCL_ERROR;
        }
        QMetaObject::invokeMethod(w,slotName,Q_ARG(int,arg0),Q_ARG(int,arg1));
        return TCL_OK;
      } else {
        continue;
      }
    }
  }
  if(slotMatched) {
    Tcl_AppendResult(interp,"signature not matched",0);
  } else {
    Tcl_AppendResult(interp,"slot '",slotName,"' not found",0);
  }
  return TCL_ERROR;
}

static const char *objectMethodNames[] = {
 "configure","confself","get","set","destroy","name",
 "class","inherits","register",
 "parent","children","query",
 "slots","signals","properties","methods",
 "event","connect","disconnect","bind",
 "enumerators","enum"
};

int QTclObjectMethods(ClientData clientData,Tcl_Interp *interp,int argc,char *argv[]) {
QObject *w=(QObject*)clientData;
const QMetaObject *m=w->metaObject();
  if(argc<2) {
    Tcl_AppendArgcArgv(interp,QTCLARSIZE(objectMethodNames),objectMethodNames);
    for(int i=0;i<w->metaObject()->methodCount();i++) {
      QMetaMethod me=m->method(i);
      if(me.methodType()==QMetaMethod::Slot) {
        Tcl_AppendElement(interp,me.signature());
      }
    }
    return TCL_OK;
  }
  if(!strcmp(argv[1],"configure")) {
    switch(argc) { 
      case 2: return QTclAppendProperties(interp,w);
      case 3: return QTclAppendPropery(interp,w,argv[2]);
      default: return QTclSetProperties(interp,w,argc-2,argv+2);
    }
  } else if((!strcmp(argv[1],"cget")) || (!strcmp(argv[1],"get"))) {
     switch(argc) {
       case 2: {
         for(int i=0;i<m->propertyCount();i++) {
           QMetaProperty mp=m->property(i);
           Tcl_AppendResult(interp,"{",0);
           Tcl_AppendElement(interp,mp.name());
           if(QTclAppendPropertyValue(interp,w,mp.name(),1)!=TCL_OK) {
             Tcl_ResetResult(interp);
             Tcl_AppendResult(interp,"error in getall",0);
             return TCL_ERROR;
           }
           Tcl_AppendResult(interp,"}",0);
         }
         break;
       }
       case 3:
         return QTclAppendPropertyValue(interp,w,argv[2]);
       default:
         return Tcl_WrongArgs(interp,1,argv,"get <property>");
     }
  } else if(!strcmp(argv[1],"set")) {
     if(argc<4) {
       Tcl_WrongArgs(interp,2,argv,"set [<property> <value>]+");
       return TCL_ERROR;
     }
     return QTclSetProperties(interp,w,argc-2,argv+2);
  } else if(!strcmp(argv[1],"destroy")) {
    if(argc!=2) Tcl_WrongArgs(interp,2,argv,0);
    delete w;
    return TCL_OK;
  } else if(!strcmp(argv[1],"name")) {
    if(argc!=2) Tcl_WrongArgs(interp,2,argv,0);
     QTclInterp::appendName(interp,w);
     return TCL_OK;
  } else if(!strcmp(argv[1],"class")) {
    Tcl_AppendElement(interp,m->className());
    return TCL_OK;
  } else if(!strcmp(argv[1],"classes")) {
    while(m) {
      Tcl_AppendElement(interp,m->className());
      m=m->superClass();
    }
    return TCL_OK;
  } else if(!strcmp(argv[1],"inherits")) {
    if(argc!=3) return Tcl_WrongArgs(interp,2,argv,"<className>");
    Tcl_AppendBool(interp,w->inherits(argv[2]));
    return TCL_OK;
  } else if(!strcmp(argv[1],"parent")) {
    if(argc!=2) Tcl_WrongArgs(interp,2,argv,0);
    QTclInterp::appendName(interp,w->parent());
    return TCL_OK;
  } else if(argc==3 && !strcmp(argv[1],"tr")) {
    Tcl_AppendResult(interp,QTclS2C(w->tr(argv[2])),0);
    return TCL_OK;
  } else if(!strcmp(argv[1],"children")) {
    int deepMode=0;
    const char *className=0;
    for(int i=2;i<argc;i++) {
      if(!strcmp(argv[i],"-deep")) {
        deepMode=1;
      } else if(!strcmp(argv[i],"-class") && i+1<argc) {
        className=argv[i+1];
        i++;
      } else {
        Tcl_WrongArgs(interp,2,argv,"[-deep] [-class <className>]");
        return TCL_ERROR;
      }
    }
    QTclInterp::appendChildren(interp,w,deepMode,className);
    return TCL_OK;
  } else if(!strcmp(argv[1],"slots")) {
    for(int i=0;i<m->methodCount();i++) {
      QMetaMethod mm=m->method(i);
      if(mm.methodType()==QMetaMethod::Slot) {
        Tcl_AppendElement(interp,mm.signature());
      }
    }
    return TCL_OK;
  } else if(argc==2 && !strcmp(argv[1],"signals")) {
    for(int i=0;i<m->methodCount();i++) {
      QMetaMethod mm=m->method(i);
      if(mm.methodType()==QMetaMethod::Signal) {
        Tcl_AppendElement(interp,mm.signature());
      }
    }
    return TCL_OK;
  } else if(argc==2 && !strcmp(argv[1],"operations")) {
    for(int i=0;i<m->methodCount();i++) {
      QMetaMethod mm=m->method(i);
      if(mm.methodType()==QMetaMethod::Method) {
        Tcl_AppendElement(interp,mm.signature());
      }
    }
    return TCL_OK;
  } else if(argc==2 && !strcmp(argv[1],"properties")) {
    for(int i=0;i<m->propertyCount();i++) {
      QMetaProperty mp=m->property(i);
      Tcl_AppendElement(interp,mp.name());
    }
    return TCL_OK;
  } else if(argc==2 && !strcmp(argv[1],"enumerators")) {
    for(int i=0;i<m->enumeratorCount();i++) {
      QMetaEnum me=m->enumerator(i);
      Tcl_AppendElement(interp,me.name());
    }
    return TCL_OK;
  } else if(!strcmp(argv[1],"methods")) {
    int mode;
    if(scanSelf(interp,argc,argv,&mode)!=TCL_OK) return TCL_ERROR;
    Tcl_CmdProc *proc=QTclInterp::findMethods(w);
    return proc((ClientData)w,interp,mode?1:0,argv);//recurse for methods
  } else if(!strcmp(argv[1],"connect")) {
    if(argc!=5) return Tcl_WrongArgs(interp,2,argv,"<signal> <widget> <slot>");
    QObject *other;
    if(QTclInterp::getObjectByName(interp,argv[3],&other)!=TCL_OK) return TCL_ERROR;
    QString si("2"); si.append(argv[2]);
    QString sl("1"); sl.append(argv[4]);
    if(!QObject::connect(w,si.toAscii().data(),other,sl.toAscii().data())) {
      Tcl_AppendResult(interp,"connect failed",0);
      return TCL_ERROR;
    }
    return TCL_OK;
  } else if(!strcmp(argv[1],"disconnect")) {
    switch(argc) {
      case 2:
        w->disconnect();
        break;
      case 3: {
        QString si("2"); si.append(argv[2]);
        w->disconnect(si.toAscii().data());
        }
        break;
      default:
      return Tcl_WrongArgs(interp,2,argv,"[<signal>]");
    }
  } else if(!strcmp(argv[1],"bind")) {
    switch(argc) {
      case 2:
        return QTclBinding::list(interp,w);
        break;
      case 3:
        return QTclBinding::info(interp,w,argv[2]);
      default:
        if(argc==4 && argv[3][0]==0) {
          return QTclBinding::unbind(interp,w,argv[2]);
        }
        return QTclBinding::bind(interp,w,argv[2],argc-3,argv+3);
    }
  } else if(!strcmp(argv[1],"event")) {
    return QTclEvent::event(interp,w,argc-2,argv+2);
  } else if(!strcmp(argv[1],"register") ) {
    int deepMode=0;
    switch(argc) {
      case 2:
        break;
      case 3:
        if(!strcmp(argv[2],"-deep")) deepMode=1;
        break;
      default:
        return Tcl_WrongArgs(interp,2,argv,"[-deep]");
    }
    QTclInterp::registerObject(interp,w,deepMode);
    return TCL_OK;
  } else if(!strcmp(argv[1],"enum") ) {
    switch(argc) {
      default: 
        return Tcl_WrongArgs(interp,2,argv,"<enumType> <value>");
      case 3: {
        int eid=w->metaObject()->indexOfEnumerator(argv[2]);
        if(eid<0) {
          Tcl_AppendResult(interp,"enum ",argv[2]," not found",0);
          return TCL_ERROR;
        }
        const QMetaEnum e=m->enumerator(eid);
        Tcl_AppendResult(interp,e.name(),0);
        for(int i=0;i<e.keyCount();i++) {
          Tcl_AppendElement(interp,e.key(i));
          Tcl_AppendInt(interp,e.value(i));
        }
        return TCL_OK;
      }
    }
  } else if(QTclInvokeSlot(interp,w,argv[1],argc-2,argv+2)!=TCL_OK) {
    Tcl_WrongArgs(interp,1,argv,0);
    Tcl_CmdProc *proc=QTclInterp::findMethods(w);
    proc((ClientData)w,interp,1,argv);
    return TCL_ERROR;
  }
  return TCL_OK;
}

int QTclObjectClassMethods(ClientData clientData,Tcl_Interp *interp,int argc,char *argv[]) {
QTclInterp *qinterp=(QTclInterp*)clientData;
  if(argc<2) return Tcl_WrongArgs(qinterp->interp(),1,argv,"tr");
  if(argc==3 && !strcmp(argv[1],"tr")) {
    Tcl_AppendResult(interp,QTclS2C(QObject::tr(argv[2])));
    return TCL_OK;
  }
  return Tcl_WrongArgs(interp,1,argv,"tr");
}

int QTclObjectInit(QTclInterp * qinterp) {
  QTclInterp::registerMethods("QObject",&QTclObjectMethods);
  QTclSizePolicyInit(qinterp);
  return TCL_OK;
}
