extern "C" {
#include <tcl.h>
}

#include <QMap>

#include "QTclUtil.h"
#include "QTclPixmap.h"
#include "QTclInterp.h"

static QMap<int,QString> pixmapNameMap; 

//------------------ Factory for Pixmaps (Global) ---------------------------

class QTclGlobalPixmapFactory {
public:
  static QPixmap create(const QString &name);
  static QStringList names();
  static QString name(const QPixmap pixmap);
  static void addPixmapFactory(QTclPixmapFactory *factory);
private:
  static QList<QTclPixmapFactory*> factoryList;
};

QList<QTclPixmapFactory*> QTclGlobalPixmapFactory::factoryList;

QPixmap QTclGlobalPixmapFactory::create(const QString& name) {
  for(int i=0;i<(int)factoryList.count();i++) {
    QTclPixmapFactory *f=factoryList.at(i);
    QPixmap result=f->create(name);
    if(!result.isNull()) {
      pixmapNameMap[result.serialNumber()]=name;
      return result;
    }
  }
  return QPixmap();
}

QStringList QTclGlobalPixmapFactory::names() {
QStringList result;
  for(int i=0;i<(int)factoryList.count();i++) {
    QTclPixmapFactory *f=factoryList.at(i);
    result+=f->names();
  }
  return result;
}

QString QTclGlobalPixmapFactory::name(const QPixmap pixmap) {
  for(int i=0;i<(int)factoryList.count();i++) {
    QTclPixmapFactory *f=factoryList.at(i);
    QString name=f->name(pixmap);
    if(name.length()) return name;
  }
  return pixmapNameMap[pixmap.serialNumber()];
}

void QTclGlobalPixmapFactory::addPixmapFactory(QTclPixmapFactory *factory) {
  factoryList.append(factory);
}

//------------------ Factory for Qt-Pixmaps -------------------------------

class QTclQtPixmapFactory : public QTclPixmapFactory {
public:
  virtual QPixmap create(const QString &name) {
    return QPixmap(name);
  }
  virtual QStringList names() {
    return QStringList();
  }
  virtual QString name(QPixmap /*pixmap*/) {
    return QString();
  }
  virtual ~QTclQtPixmapFactory() {}
};

// ---------------------- Tcl-Interfacce -----------------------------------

int QTclAppendPixmap(Tcl_Interp *interp,const QPixmap pixmap) {
  if(pixmap.isNull()) {
    Tcl_AppendElement(interp,"");
    return TCL_OK;
  }
  QString name=QTclGlobalPixmapFactory::name(pixmap);
  if(name.length()) {
    Tcl_AppendElement(interp,QTclS2C(name));
    return TCL_OK;
  }
  char buffer[200];
  sprintf(buffer,"QP%8.8x",(unsigned int)(pixmap.serialNumber()));
  Tcl_AppendElement(interp,buffer);
  return TCL_OK;
}

int QTclAppendPixmapPtr(Tcl_Interp *interp,const QPixmap *pixmapPtr) {
  if(!pixmapPtr) {
    Tcl_AppendElement(interp,"");
    return TCL_OK;
  }
  return QTclAppendPixmap(interp,*pixmapPtr);
}

int QTclGetPixmap(Tcl_Interp *interp,const QString &name,QPixmap *pp) {
  *pp=QTclGlobalPixmapFactory::create(name);
  if(!(pp->isNull())) return TCL_OK;
  Tcl_AppendResult(interp,"pixmap '",QTclS2C(name),"' not found",0);
  return TCL_ERROR;
}

int QTclPixmapClassMethods(ClientData clientData,Tcl_Interp *interp,int argc,char *argv[]) {
//QTclInterp *qinterp=(QTclInterp*)clientData;
  if(argc<2) {
    Tcl_AppendElement(interp,"names");
    if(argc<1) return TCL_OK;
  }
  if(argc==2 && !strcmp(argv[1],"names") ) {
    Tcl_AppendQStringList(interp,QTclGlobalPixmapFactory::names());
    return TCL_OK;
  }
  Tcl_WrongArgs(interp,1,argv," ");
  QTclPixmapClassMethods(clientData,interp,0,argv);
  return TCL_ERROR;
}

//---------------------- InitModule Code ------------------------------------

int QTclInitPixmap(QTclInterp * /*qinterp*/) {
  QTclGlobalPixmapFactory::addPixmapFactory(new QTclQtPixmapFactory);
  QTclInterp::registerClassMethods("QPixmap",&QTclPixmapClassMethods);
  return TCL_OK;
}


