#include <vector>
#include <cmath>

using namespace std;

#define QTCLSQR(x) ((x)*(x))

class Transform {
public:
  double A,B,C,D,E,F,G,H,J;
};

class Vector {
public:
  Vector() {} // uninitialized !!!
  Vector(double x,double y,double z): _x(x),_y(y),_z(z) {}
  Vector(const Vector &v2) : _x(v2._x),_y(v2._y),_z(v2._z) {}
  void init() { _x=_y=_z=0.0; }
  double x() const { return _x; }
  double y() const { return _y; }
  double z() const { return _z; }
  void setX(double x) { _x=x; }
  void setY(double y) { _y=y; }
  void setZ(double z) { _z=z; }
  Vector& operator=(const Vector&v2) { _x=v2._x; _y=v2._y; _z=v2._z; return *this; }
  Vector& operator-=(const Vector &v2) { _x-=v2._x; _y-=v2._y; _z-=v2._z; return *this; }
  Vector& operator+=(const Vector &v2) { _x+=v2._x; _y+=v2._y; _z+=v2._z; return *this; }
  Vector operator-(const Vector &v2) const { return Vector(_x-v2._x,_y-v2._y,_z-v2._z); }
  void normalize() { double len=sqrt(QTCLSQR(_x)+QTCLSQR(_y)+QTCLSQR(_z)); if(len!=0) { _x/=len; _y/=len; _z/=len; } }
  Vector spat(const Vector &v2) const { return Vector(_y*v2._z-_z*v2._y,_z*v2._x-_x*v2._z,_x*v2._y-_y*v2._x); }
  double scalar(const Vector &v2) { return _x*v2._x+_y*v2._y+_z*v2._z; }
  Vector transform(const Transform &t) {
    return Vector(
      _x*t.A+_y*t.B+_z*t.C,
      _x*t.D+_y*t.E+_z*t.F,
      _x*t.G+_y*t.H+_y*t.J
    );
  }
  Vector project(const Vector &v) const {
    double s=z()+v.z();
    if(s==0) return v;
    return Vector(
      v.x()-v.z()*(_x-v.x())/s,
      v.y()-v.z()*(_y-v.y())/s,
      v.z()-v.z()*(_z-v.z())/s
    );
  }
private:
  double _x;
  double _y;
  double _z;
};


class Camera : public Vector {
public:
  Camera(): Vector() {}
  Camera(double x,double y,double z): Vector(x,y,z) {}
  virtual ~Camera() {}
  virtual double rotX() const =0;
  virtual double rotY() const =0;
  virtual double rotZ() const =0;
  virtual double scaleX() const =0;
  virtual double scaleY() const =0;
  virtual double scaleZ() const =0;
};


class Triangle;

class Vertex : public Vector {
public:
  Vertex(double _x,double _y,double _z):Vector(_x,_y,_z),_normalValid(false) {}
  void addUsedBy(Triangle *t) {
    usedBy.push_back(t);
    _normalValid=false;
  }
  Vector* normal() { 
    if(!_normalValid) {
      recalcNormal();
      _normalValid=true;
    }
    return &_normalCache; 
  }
private:
  void recalcNormal();
  vector <Triangle*> usedBy;
  Vector _normalCache; 
  bool _normalValid;
};

class Triangle {
public:
  Triangle(Vertex *v1,Vertex *v2,Vertex *v3):_v1(v1),_v2(v2),_v3(v3) {
    calcNormal();
    _v1->addUsedBy(this);
    _v2->addUsedBy(this);
    _v3->addUsedBy(this);
  }
  const Vector &normal() const { return _normal; }
  Vertex *v1() { return _v1; }
  Vertex *v2() { return _v2; }
  Vertex *v3() { return _v3; }
private:
  Vertex *_v1,*_v2,*_v3;
  Vector _normal;
  void calcNormal() {
    _normal.init();
    _normal+=*_v1;
    _normal+=*_v2;
    _normal+=*_v3;
    _normal.normalize();
  }
};

class TriangleVisitor {
public:
  virtual void visit(Triangle *,bool visible,const Vector &v1,const Vector &v2,const Vector &v3) const =0;
  virtual ~TriangleVisitor() {}
};

class World {
public:
  void addVertex(double x,double y,double z) {
    vertices.push_back(new Vertex(x,y,z));
  }
  void addTriangle(int v1,int v2,int v3) {
    triangles.push_back(new Triangle(vertices[v1],vertices[v2],vertices[v3]));
  }
  ~World() {
    for(vector <Triangle*>::const_iterator triangle=triangles.begin();triangle!=triangles.end();++triangle) {
      delete (*triangle);
    }
    for(vector <Vertex*>::const_iterator vertex=vertices.begin();vertex!=vertices.end();++vertex) {
      delete (*vertex);
    }
    triangles.clear();
    vertices.clear();
  }
  void visitCamera(Camera *c,TriangleVisitor *visitor) {
    double six=sin(c->rotX());
    double siy=sin(c->rotY());
    double siz=sin(c->rotZ());
    double cox=cos(c->rotX());
    double coy=cos(c->rotY());
    double coz=cos(c->rotZ());
    Transform t;
    t.A=coy*coz*c->scaleX();
    t.B=coy*siz*c->scaleY();
    t.C=-siy*c->scaleZ();
    t.D=(six*siy*coz-cox*siz)*c->scaleX();
    t.E=(six*siy*siz+cox*coz)*c->scaleY();
    t.F=(six*coy)*c->scaleZ();
    t.G=(cox*siy*coz+six*siz)*c->scaleX();
    t.H=(cox*siy*siz-six*coz)*c->scaleY();
    t.J=(cox*coy)*c->scaleZ();
    for(vector<Triangle*>::const_iterator i=triangles.begin();i!=triangles.end();++i) {
      Triangle *triangle=(*i);
      Vector t1=triangle->v1()->transform(t);
      Vector t2=triangle->v2()->transform(t);
      Vector t3=triangle->v3()->transform(t);
      Vector normal=(t2-t1).spat(t3-t1);
      Vector s(t1);
      s-=*c;
      bool visible=(s.scalar(normal))>0;
      Vector p1=t1.project(*c);
      Vector p2=t2.project(*c);
      Vector p3=t3.project(*c);
      visitor->visit(triangle,visible,p1,p2,p3);
    }
  }
  void makeCubus(double x,double y,double z,double scale);
  void makeDolphin(double x,double y,double z,double scale);
private:
  vector <Triangle*> triangles;
  vector <Vertex*>   vertices;
};

