extern "C" {
#include <math.h>
};

#include <QtCore/Qt>
#include <QtCore/QEvent>
#include <QtGui/QMouseEvent>
#include <QtGui/QPainter>
#include <QtGui/QPixmap>
#include <QtGui/QPolygon>

#include "QTcl3D.h"
#include "QTclAngleControl.h"

#define QTCLARSIZE(a) ((int)(sizeof(a)/sizeof(a[0])))

QTclAngleControl::QTclAngleControl(QWidget *parent):QWidget(parent),_alpha(0),_phi(0),_theta(0){
}

//------------------- Controller ------------------------------------------------

void QTclAngleControl::setAlpha(double alpha) {
  while(alpha<0)       { alpha+=2*M_PI; }
  while(alpha>=2*M_PI) { alpha-=2*M_PI; }
  _alpha=alpha;
  repaint();
  emit changed();
}

void QTclAngleControl::setPhi(double phi) {
  while(phi<-M_PI) { phi+=2*M_PI; }
  while(phi> M_PI) { phi-=2*M_PI; }
  _phi=phi;
  repaint();
  emit changed();
}

void QTclAngleControl::setTheta(double theta) {
  while(theta<0)     { theta+=2*M_PI; }
  while(theta>2*M_PI) { theta-=2*M_PI; }
  _theta=theta;
  repaint();
  emit changed();
}

void QTclAngleControl::mousePressEvent(QMouseEvent *e) {
  clickPos=e->pos();
}

void QTclAngleControl::mouseMoveEvent(QMouseEvent *e) {
  double deltax=(e->pos().x()-clickPos.x())*2.0*M_PI/width();
  double deltay=(e->pos().y()-clickPos.y())*2.0*M_PI/height();
  setAlpha(alpha()+deltax);
  setPhi(phi()+deltay);
  clickPos=e->pos();
}

void QTclAngleControl::wheelEvent(QWheelEvent *e) {
  double deltaw=(e->delta()/120.0)*2*M_PI/72;
  setTheta(theta()+deltaw);
}

//------------------- View ------------------------------------------------

static void drawTriangle(QPainter &p,int x1,int y1,int x2,int y2,int x3,int y3){
QPolygon a(3);
  a.setPoint(0,x1,y1);
  a.setPoint(1,x2,y2);
  a.setPoint(2,x3,y3);
  p.drawPolygon(a);
}


class AngleCamera:public Camera { 
public:
  AngleCamera(double x,double y,double z):Camera(x,y,z),_rotx(0.0),_roty(0.0),_rotz(0.0),_scalex(1.0),_scaley(1.0),_scalez(1.0) {}
  virtual double rotX() const { return _rotx; }
  virtual double rotY() const { return _roty; }
  virtual double rotZ() const { return _rotz; }
  virtual double scaleX() const { return _scalex; }
  virtual double scaleY() const { return _scaley; }
  virtual double scaleZ() const { return _scalez; }
  virtual void setRotX(double v) { _rotx=v; update(); }
  virtual void setRotY(double v) { _roty=v; update(); }
  virtual void setRotZ(double v) { _rotz=v; update(); }
  virtual void setScaleX(double v) { _scalex=v; update(); }
  virtual void setScaleY(double v) { _scaley=v; update(); }
  virtual void setScaleZ(double v) { _scalez=v; update(); }
  virtual void update() const {}
private:
  double _rotx;
  double _roty;
  double _rotz;
  double _scalex;
  double _scaley;
  double _scalez;
};

class PainterTriangleVisitor : public TriangleVisitor {
public:
  PainterTriangleVisitor(QPainter &painter,int x,int y):_painter(painter),_x(x),_y(y) {}
  void visit(Triangle *t,bool visible,const Vector &p1,const Vector &p2,const Vector &p3) const {
    if(!visible) return;
    int x1=(int)(p1.x());
    int y1=(int)(p1.y());
    int x2=(int)(p2.x());
    int y2=(int)(p2.y());
    int x3=(int)(p3.x());
    int y3=(int)(p3.y());
    Vector lamp(-1.0,-1.0,-1.0);
    lamp.normalize();
    double gray=lamp.scalar(t->normal());
    //_painter.drawText(x1,y1,QString::number(gray));
    _painter.setPen(Qt::NoPen);
    _painter.setBrush(QColor::fromHsv(-1,(int)(100+gray*100),(int)(100.0+gray*100)));
    drawTriangle(_painter,x1,y1,x2,y2,x3,y3);
  }
private:
  QPainter &_painter;
  int _x;
  int _y;
  int _delta;
};


void QTclAngleControl::paintEvent(QPaintEvent *) {
  QPainter p(this);
  QPixmap doubleBuffer(width(),height());
  doubleBuffer.fill(palette().color(QPalette::Background));
  QPainter q(&doubleBuffer);
  q.setBrush(Qt::NoBrush);
  q.setPen(Qt::SolidLine);
  q.drawText(10,20,QString::number(_alpha));
  q.drawText(10,35,QString::number(_phi));
  q.drawText(10,50,QString::number(_theta));
  q.translate(width()/2,height()/2);
  AngleCamera camera(0.0,0.0,10000);
  camera.setRotX(_phi);
  camera.setRotY(_alpha);
  camera.setRotZ(_theta);
  camera.setScaleX(1.0);
  camera.setScaleY(1.0);
  camera.setScaleZ(1.0);
  World world;
  //world.makeCubus(0.0,0.0,0.0,QMIN(height(),width())/4.0);
  world.makeDolphin(0.0,0.0,0.0,3.0);
  PainterTriangleVisitor visitor(q,clickPos.x()-width()/2,clickPos.y()-height()/2);
  world.visitCamera(&camera,&visitor);
  p.drawPixmap(0,0,doubleBuffer);
}

QSizePolicy QTclAngleControl::sizePolicy() const {
  return QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );
}

#include "moc_QTclAngleControl.cpp"

