#include <QtCore/QFile>
#include <QtCore/QTextStream>

#include "QTclDOMTreeView.h"

#define DEFAULTDEPTH     (5)
#define DEFAULTTEXTLIMIT (100)

QTclDOMTreeView::QTclDOMTreeView(QWidget *parent)
  :QTreeWidget(parent),_textLimit(DEFAULTTEXTLIMIT),_expansionDepth(DEFAULTDEPTH),
    _pure(true), _showAttributes(true){
  setSortingEnabled(false);
  //setRootIsDecorated(true);
  //setShowSortIndicator(false);
}

QTclDOMTreeView::~QTclDOMTreeView() {
}

QTreeWidgetItem *QTclDOMTreeView::startElement(const QDomElement element,QTreeWidgetItem*parent) {
QString text;
QTreeWidgetItem *curItem=parent?new QTreeWidgetItem(parent):new QTreeWidgetItem(this);
  if (!_pure) text+="<";
  text+=element.nodeName();
  if(_showAttributes) {
    QString attributes;
    QDomNamedNodeMap attrs = element.attributes();
    unsigned long lmap = attrs.length();
    for(unsigned int j=0;j<lmap;j++) {
      QDomAttr attr=attrs.item(j).toAttr(); 
      attributes += " " + attr.name() + "=\"" + attr.value() + "\"";
    }
    text+=attributes/*.simplifyWhiteSpace()*/;
  }
  if(!_pure) text += ">";
  curItem->setText(0, text);
  return curItem;
}

QTreeWidgetItem *QTclDOMTreeView::endElement(const QDomElement element,QTreeWidgetItem*parent,QTreeWidgetItem*after) {
  if(_pure) return 0;
QTreeWidgetItem *curItem;
  curItem=parent?new QTreeWidgetItem(parent,after):new QTreeWidgetItem(this,after);
  curItem->setText(0,QString("</")+element.nodeName()+">");
  return curItem;
}

QTreeWidgetItem *QTclDOMTreeView::textNode(QDomNode &node,QTreeWidgetItem*parent) {
QString text;
  text = "'" + node.nodeValue() + "'";
  QTreeWidgetItem *curItem=0;
  QString txt;
  for(QTextStream ts(&text,QIODevice::ReadOnly);!((txt=ts.readLine()).isNull());) {
    if(_textLimit && txt.length()>_textLimit) {
      txt.truncate(_textLimit-3);
      txt.append(QString("..'"));
    }
    if(!curItem) {
      curItem=new QTreeWidgetItem(parent);
    } else {
      curItem=new QTreeWidgetItem(parent,curItem);
    }
    curItem->setText(0,txt);
  }
  return curItem;
}

void QTclDOMTreeView::showRecursive(QTreeWidgetItem *parent,QDomNode node, int depth) {
  if(!node.isElement()) {
    textNode(node,parent);
    return;
  }
  QTreeWidgetItem *curItem=startElement(node.toElement(),parent);
  if(curItem) if(depth<_expansionDepth) collapseItem(curItem); else expandItem(curItem);
  for(QDomNode child=node.lastChild();!child.isNull();child = child.previousSibling()) {
    showRecursive(curItem,child,depth+1);
  }
  curItem=endElement(node.toElement(),parent,curItem);
  if(curItem) if(depth<_expansionDepth) collapseItem(curItem); else expandItem(curItem);
}

void QTclDOMTreeView::refresh() {
  clear();
  showRecursive((QTreeWidgetItem*)0,_document.documentElement(),0);
}

bool QTclDOMTreeView::searchRecursive(QTreeWidgetItem* cur_item, const QString& searchText, bool caseSensitive,bool all) {
int result=0;
  if(!cur_item) {
    if(!(cur_item=topLevelItem(0))) return result;
  }
  const QString text(cur_item->text(0));
  if(text.contains(searchText,caseSensitive?Qt::CaseSensitive:Qt::CaseInsensitive)>0) {
    setCurrentItem(cur_item);
    scrollToItem(cur_item);
    result=1;
    if(!all) return result;
  }
  for(int i=0;i<cur_item->childCount();++i) {
    QTreeWidgetItem*child=cur_item->child(i);
    result=result || searchRecursive(child,searchText,caseSensitive,all);
    if(result && !all) break;
  }
  return result;
}

bool QTclDOMTreeView::search(const QString&text) {
  return searchRecursive(0,text,true,true);
}

bool QTclDOMTreeView::searchCase(const QString&text) {
  return searchRecursive(0,text,false,true);
}

int  QTclDOMTreeView::expansionDepth() const {
  return _expansionDepth;
}

void QTclDOMTreeView::setExpansionDepth(int newDepth) {
  if(_expansionDepth!=newDepth) {
    _expansionDepth=newDepth;
    adjustDepth(topLevelItem(0),0);
  }
}

void QTclDOMTreeView::unsetExpansionDepth() {
  setExpansionDepth(DEFAULTDEPTH);
}

void QTclDOMTreeView::adjustDepth(QTreeWidgetItem *item,int newDepth) {
  if(_expansionDepth>newDepth) {
    expandItem(item);
  } else {
    collapseItem(item);
  }
  for(int i=0;i<item->childCount();++i) {
    adjustDepth(item->child(i),newDepth+1);
  }
}

void QTclDOMTreeView::setPure(bool b) {
  if(_pure!=b) {
    _pure = b;
    refresh();
  }
}

void QTclDOMTreeView::unsetPure() {
  setPure(FALSE);
}

bool QTclDOMTreeView::pure() const {
  return _pure;
}

void QTclDOMTreeView::setShowAttributes(bool b) {
  if(_showAttributes!=b) {
    _showAttributes = b;
    refresh();
  }
}

void QTclDOMTreeView::unsetShowAttributes() {
  setShowAttributes(FALSE);
}

bool QTclDOMTreeView::showAttributes() const {
  return _showAttributes;
}

void QTclDOMTreeView::unsetDocument() {
  _document.clear();
  _filename="";
  refresh();
}

void QTclDOMTreeView::setFilename(const QString&fname) {
  QFile f(fname);
  if (!f.open(QIODevice::ReadOnly) || !_document.setContent(&f)){
    unsetDocument();
    refresh();
  }
  if(f.isOpen()) f.close();
  _filename=fname;
  refresh();
}

const QString& QTclDOMTreeView::filename() const {
  return _filename;
}

void QTclDOMTreeView::setText(const QString &text) {
  _document.setContent(text);
  _filename="";
  refresh();
}

QString QTclDOMTreeView::text() const {
  return _document.toString();
}

int QTclDOMTreeView::textLimit() const {
  return _textLimit;
}

void QTclDOMTreeView::setTextLimit(int limit) {
  if(limit==_textLimit) return;
  _textLimit=limit;
  refresh();
}

void QTclDOMTreeView::unsetTextLimit() {
  setTextLimit(DEFAULTTEXTLIMIT);
}

#include "moc_QTclDOMTreeView.cpp"
